import { Menu, MenuButton, MenuItem, MenuItems } from "@headlessui/react";
import { IoIosArrowDown } from "react-icons/io";
import { useAppDispatch, useAppSelector } from "../../../../utils/hooks";
import { LuTrash2 } from "react-icons/lu";
import { BsReply } from "react-icons/bs";
import { RiArrowGoForwardLine } from "react-icons/ri";
import {
  addMessage,
  updateMessageOptions,
} from "../../../../store/Slices/MessageOptionsSlice";
import { Message } from "../../../../types/MessageListType";
import useApiPost from "../../../../hooks/PostData";
import { updateIsStarMessage } from "../../../../store/Slices/MessageListSlice";
import toast from "react-hot-toast";
import { updateSendMessageData } from "../../../../store/Slices/SendMessageSlice";
import { useStarMessageList } from "../../../../store/api/useStarMessageList";
import { useTranslateText } from "../../../../hooks/useTranslateText";
import PinMessageButton from "./PinMessageButton";
import { FaRegStar } from "react-icons/fa6";

export default function SelectedMessageOption({
  messageData,
}: {
  messageData: Message;
}) {
  let dispatch = useAppDispatch();
  let currentConversation = useAppSelector(
    (state) => state.CurrentConversation,
  );

  const userData = useAppSelector((state) => state.userData);
  const { postData } = useApiPost();
  const { refetch } = useStarMessageList();
  const translate = useTranslateText();

  async function starMessage(message_id: number, remove_from_star: boolean) {
    await postData("chat/star-unstar-message", {
      message_id: String(message_id),
    });

    // Refetch the star message ==================================================================================
    refetch();
    toast.success(
      `Message  ${remove_from_star ? "removed from" : "added to"} star  `,
    );

    dispatch(
      updateIsStarMessage({
        message_id,
        is_star_message: remove_from_star ? false : true,
      }),
    );
  }

  return (
    <>
      <div
        className={`relative text-darkText ${messageData.User?.user_id == userData?.user_id ? "" : "text-darkText"} `}
      >
        <Menu>
          <MenuButton
            onClick={(event) => {
              event.stopPropagation(); // Prevents the parent onClick event
              // setSelectedConversation(e);
            }}
            className="invisible items-center font-semibold shadow-2xl focus:outline-none group-hover:visible data-[focus]:outline-0"
          >
            <IoIosArrowDown className="cursor-pointer text-xl drop-shadow-[0_0_1px_white]" />
          </MenuButton>

          <MenuItems
            transition
            anchor={messageData.User?.user_id == userData?.user_id ? "bottom end" : "bottom start"}
            className="z-10 w-52 origin-top-right rounded-xl border border-borderColor bg-modalBg p-1 text-sm/6 transition duration-200 ease-out [--anchor-gap:var(--spacing-1)] focus:outline-none data-[closed]:scale-95 data-[closed]:opacity-0"
          >
            <MenuItem>
              <button
                onClick={() => {
                  dispatch(
                    updateMessageOptions({
                      selectMessage: true,
                      delete_message: true,
                      // show only delete from button if message is allready deleted ======================================================
                      delete_only_from_me:
                        messageData.deleted_for_everyone ||
                        messageData.deleted_for.includes(
                          userData?.user_id.toString(),
                        ) ||
                        messageData.User?.user_id != userData?.user_id
                          ? true
                          : false,
                    }),
                  );
                  dispatch(addMessage(messageData));
                }}
                className="group flex w-full items-center justify-between gap-2 rounded-lg px-4 py-1.5 data-[focus]:bg-dropdownOptionHover"
              >
                <span>{translate("Delete")}</span>
                <LuTrash2 className="text-xl" />
              </button>
            </MenuItem>
            <MenuItem>
              <button
                onClick={() =>
                  messageData.starred
                    ? starMessage(messageData.message_id, true)
                    : starMessage(messageData.message_id, false)
                }
                className="group flex w-full items-center justify-between gap-2 rounded-lg px-4 py-1.5 data-[focus]:bg-dropdownOptionHover"
              >
                <span>
                  {translate(messageData.starred ? "Unstar" : "Star")}
                </span>
                <FaRegStar className="text-xl" />
              </button>
            </MenuItem>
            <MenuItem>
              <button
                onClick={() => {
                  dispatch(
                    updateSendMessageData({
                      reply_to: messageData.message_id,
                    }),
                  );
                }}
                className="group flex w-full items-center justify-between gap-2 rounded-lg px-4 py-1.5 data-[focus]:bg-dropdownOptionHover"
              >
                <span>{translate("Reply")}</span>
                <BsReply className="text-xl" />
              </button>
            </MenuItem>
            <MenuItem>
              <button
                onClick={() => {
                  dispatch(
                    updateMessageOptions({
                      selectMessage: true,
                      forward_message: true,
                    }),
                  );
                  dispatch(addMessage(messageData));
                }}
                className="group flex w-full items-center justify-between gap-2 rounded-lg px-4 py-1.5 data-[focus]:bg-dropdownOptionHover"
              >
                <span>{translate("Forward")}</span>
                <RiArrowGoForwardLine className="text-xl" />
              </button>
            </MenuItem>
            <MenuItem>
              <button className="w-full rounded-lg  data-[focus]:bg-dropdownOptionHover">
                <PinMessageButton messageData={messageData} />
              </button>
            </MenuItem>
          </MenuItems>
        </Menu>
      </div>
    </>
  );
}
