import { formatRelative } from "date-fns";
import { StoryListRes, MyStory, RecentStory } from "../../../../types/StatusTypes";
import { UserData } from "../../../../types/UserDataType";


interface UserStories {
  user: {
    name: string;
    profilePic: string;
  };
  stories: {
    url: string;
    type?: "image" | "video";
    header?: { heading: string; subheading: string; profileImage: string };
    duration?: number;
    statusId: number;
    mediaId?: number;
    user_id: number;
    status_text: string;
  }[];
}

export function mapStoriesGrouped(res: StoryListRes, userData: UserData): UserStories[] {
  if (!res || !res.data) return [];

  const { my_stories, recent_stories, viewed_stories } = res.data;
  const grouped: UserStories[] = [];

  // ✅ My Stories
  if (my_stories && my_stories.length > 0) {
    const myUser = {
      name: "My Stories",
      profilePic: userData.profile_pic,
    };

    const myStories = my_stories.map((story: MyStory) => {
      const isVideo =
        story.story_type === "video" || story.media.endsWith(".mp4");
      return {
        url: story.media,
        type: isVideo ? ("video" as const) : ("image" as const),
        header: {
          heading: myUser.name,
          subheading: formatRelative(new Date(story.updatedAt), new Date()),
          profileImage: myUser.profilePic,
        },
        duration: isVideo ? undefined : 5000,
        statusId: story.story_id,
        user_id: story.user_id,
        status_text: story.caption,
      };
    });

    grouped.push({ user: myUser, stories: myStories });
  }

  // ✅ Recent Stories (not viewed)
  if (recent_stories && recent_stories.length > 0) {
    recent_stories.forEach((user: RecentStory) => {
      const stories = (user.stories || []).map((story: any) => {
        const isVideo =
          story.story_type === "video" || story.media.endsWith(".mp4");
        return {
          url: story.media,
          type: isVideo ? ("video" as const) : ("image" as const),
          header: {
            heading: user.full_name || user.user_name || user.mobile_num,
            subheading: formatRelative(new Date(story.updatedAt), new Date()),
            profileImage: user.profile_pic,
          },
          duration: isVideo ? undefined : 5000,
          statusId: story.story_id,
          user_id: user.user_id,
          status_text: story.caption,
        };
      });

      if (stories.length > 0) {
        grouped.push({
          user: {
            name: user.full_name || user.user_name || user.mobile_num,
            profilePic: user.profile_pic,
          },
          stories,
        });
      }
    });
  }

  // ✅ Viewed Stories (already seen)
  if (viewed_stories && viewed_stories.length > 0) {
    viewed_stories.forEach((user: RecentStory) => {
      const stories = (user.stories || []).map((story: any) => {
        const isVideo =
          story.story_type === "video" || story.media.endsWith(".mp4");
        return {
          url: story.media,
          type: isVideo ? ("video" as const) : ("image" as const),
          header: {
            heading: user.full_name || user.user_name || user.mobile_num,
            subheading: formatRelative(new Date(story.updatedAt), new Date()),
            profileImage: user.profile_pic,
          },
          duration: isVideo ? undefined : 5000,
          statusId: story.story_id,
          user_id: user.user_id,
          status_text: story.caption,
        };
      });

      if (stories.length > 0) {
        grouped.push({
          user: {
            name: user.full_name || user.user_name || user.mobile_num,
            profilePic: user.profile_pic,
          },
          stories,
        });
      }
    });
  }

  return grouped;
}
